//--------------------------------------------------------------//
// In/Out Colour Correction 
//
// Copyright (c) LWKS Software Ltd.  All Rights Reserved
//--------------------------------------------------------------//
DeclareLightworksEffect( IOCC, Colour, Internal, kNoFlags, CanSize );

//--------------------------------------------------------------
// Params
//--------------------------------------------------------------
DeclareColourParam( BlackLevel,       "Black Level", kNoGroup, kNoFlags, 0,0,0,1 );
DeclareColourParam( GreyLevel,        "Grey Level",  kNoGroup, kNoFlags, 0.5,0.5,0.5,1 );
DeclareColourParam( WhiteLevel ,      "White Level", kNoGroup, kNoFlags, 1,1,1,1 );
DeclareColourParam( ShadowTintColour, "Black Level", kNoGroup, kNoFlags, 0,0,0,1 );
DeclareColourParam( MidTintColour,    "Black Level", kNoGroup, kNoFlags, 0,0,0,1 );
DeclareColourParam( HighTintColour,   "Black Level", kNoGroup, kNoFlags, 0,0,0,1 );
DeclareFloatParam( Saturation,        "Saturation",  kNoGroup, kNoFlags, 0,0,0 );
DeclareColourParam( Gamma,            "Gamma",       kNoGroup, kNoFlags, 0,0,0,0 );
DeclareColourParam( Contrast,         "Contrast",    kNoGroup, kNoFlags, 0,0,0,0 );
DeclareColourParam( Brightness,       "Brightness",  kNoGroup, kNoFlags, 0,0,0,0 );
DeclareColourParam( Gain,             "Gain",        kNoGroup, kNoFlags, 0,0,0,0 );
DeclareFloatParam( HueBias,           "Hue Bias",    kNoGroup, kNoFlags, 0,0,0 );
DeclareFloatParam( SatBias,           "Sat Bias",    kNoGroup, kNoFlags, 0,0,0 );
DeclareFloatParam( ValBias,           "Val Bias",    kNoGroup, kNoFlags, 0,0,0 );
DeclareFloatParam( SatScale,          "Sat scale",   kNoGroup, kNoFlags, 0,0,0 );
DeclareFloatParam( ValScale,          "Val scale ",  kNoGroup, kNoFlags, 0,0,0 );

DeclareColourParam( SelectionColour,    "Colour",    "Selection", kNoFlags, 0,0,0,0 );
DeclareColourParam( SelectionTolerance, "Tolerance", "Selection", kNoFlags, 0,0,0,0 );
DeclareColourParam( SelectionSoftness,  "Softness",  "Selection", kNoFlags, 0,0,0,0 );
DeclareBoolParam( SelectionInvert,      "Invert",    "Selection", false );
DeclareBoolParam( SelectionReveal,      "Reveal",    "Selection", false );

float _pi     = 3.14159;
float _pi2    = 6.2831853;
float _offset = 0.5 / 256.0;
float _scale  = 255.0 / 256.0;
float _rt3    = 0.57735; // 1.0 / sqrt( 3.0 );

float _oneSixth   = 1.0 / 6.0;
float _twoSixths  = 2.0 / 6.0;
float _oneHalf    = 0.5;
float _fourSixths = 4.0 / 6.0;
float _fiveSixths = 5.0 / 6.0;

#define HUE_IDX 0
#define SAT_IDX 1
#define VAL_IDX 2

//--------------------------------------------------------------//
// Inputs
//--------------------------------------------------------------//
DeclareInput( Input );
DeclareInput( Lut );
DeclareInput( HSL );
DeclareInput( Inside );
DeclareInput( Outside )

DeclareMask;

bool allPositive( float3 pixel )
{
   return pixel.r > 0 && pixel.g > 0 && pixel.b > 0;
}

//---------------------------------------------------------------------------------------
// Logic
//---------------------------------------------------------------------------------------
DeclareEntryPoint( RGB )
{
   // Read the source image pixel
   float4 src = ReadPixel( Input, uv1 );
   float4 hsv = tex2D( HSL, uv3 );
   float4 ret = src;

   if ( SelectionReveal == false )
   {
      // Do balancing
      float greyAverage = ( GreyLevel.r + GreyLevel.g + GreyLevel.b ) / 3.0;
      ret -= BlackLevel;
      ret /= ( WhiteLevel - BlackLevel );
      ret -= GreyLevel;
      ret += greyAverage;
      ret = clamp( ret, 0.0, 1.0 );

      // Calc its brightness (so that it can be treated as a shadow/midtone/highlight as appropriate)
      float lum = ( ret.r + ret.g + ret.b ) / 3.0;

      // Apply the shadow/midtone/highlight tints
      float2 lutPos   = float2( ( lum * _scale ) + _offset, 0.5 );
      float4 lutPixel = tex2D( Lut, lutPos );
      ret += ShadowTintColour * lutPixel.r;
      ret += MidTintColour    * lutPixel.g;
      ret += HighTintColour   * lutPixel.b;

      float avg = ( ret.r + ret.g + ret.b ) / 3.0;
      float4 avg4 = float4( avg, avg, avg, avg );
      ret = avg4 + ( ( ret - avg4 ) * Saturation );

      // Do brightness, gamma & contrast logic
      ret = ( ( ( ( pow( saturate( ret ), 1 / Gamma ) * Gain ) + Brightness ) - 0.5 ) * Contrast ) + 0.5;
   }
   else
      ret = float4( 1,1,1,1 );

   // Calc difference between current pixel and specified key-colour

   float4 diff = abs( hsv - SelectionColour );

   if ( diff[ HUE_IDX ] > 0.5 )
      diff[ HUE_IDX ] = 1.0 - diff[ HUE_IDX ];

   // Work out how transparent/opaque the corrected pixel will be

   float mixAmount = 0;

   if ( allPositive( SelectionTolerance.rgb + SelectionSoftness.rgb - diff.rgb ) )
   {
      if ( allPositive( SelectionTolerance.rgb - diff.rgb ) )
         mixAmount = 1;
      else
      {
         diff -= SelectionTolerance;
         diff /= SelectionSoftness;
         mixAmount = 1.0 - ( max( diff[ HUE_IDX ], max( diff[ SAT_IDX ], diff[ VAL_IDX ] ) ) );
      }
   }

   ret.a = lerp( mixAmount, 1.0 - mixAmount, SelectionInvert );

   return ret;
}

//--------------------------------------------------------------//
// RGBtoHSL
//--------------------------------------------------------------//
DeclareEntryPoint( RGBtoHSL )
{
   float4 hsv = float4( 0,0,0,0 );
   float4 rgba = ReadPixel( Input, uv1 );

   float maxComponentVal = max( max( rgba.r, rgba.g ), rgba.b );
   float minComponentVal = min( min( rgba.r, rgba.g ), rgba.b );
   float componentRange  = maxComponentVal - minComponentVal;

   hsv[ VAL_IDX ] = maxComponentVal;
   hsv[ SAT_IDX ] = componentRange / maxComponentVal;

   if ( hsv[ SAT_IDX ] != 0 )
   {
      if ( rgba.r == maxComponentVal )
      {
         hsv[ HUE_IDX ] = ( rgba.g - rgba.b ) / componentRange;
      }
      else if ( rgba.g == maxComponentVal )
      {
         hsv[ HUE_IDX ] = 2 + ( ( rgba.b - rgba.r ) / componentRange );
      }
      else
      {
         hsv[ HUE_IDX ] = 4 + ( ( rgba.r - rgba.g ) / componentRange );
      }

      hsv[ HUE_IDX ] *= _oneSixth;

      if ( hsv[ HUE_IDX ] < 0 )
         hsv[ HUE_IDX ] += 1.0;
   }

   return hsv;
}

//--------------------------------------------------------------
// ApplyHSVCorrections
//
// Convert to HSV, apply the corrections then convert back th RGB
//--------------------------------------------------------------
DeclareEntryPoint( ApplyHSVCorrections )
{
   float4 src = ReadPixel( Input, uv1 );
   float4 ret;

   float i  = ( src.r + src.g + src.b ) / 3.0;
   float r  = src.r;
   float g  = src.g;
   float b  = src.b;
   float rg = r - g;
   float rb = r - b;

   // Hue
   float h = acos( ( rg + rb ) / ( 2 * sqrt( rg * rg + rb * ( g - b ) ) ) ) / _pi2;

   if ( b > g )
      h = 1 - h;

   // Saturation
   float s = 1 - min( min( r, g ), b ) / i;

   // Scale and bias our components
   h = frac( h + HueBias );
   s = saturate( s * SatScale + SatBias );
   i = saturate( i * ValScale  + ValBias );

   float h3 = 3 * h;
   float x = (2 * floor(h3) + 1) / 6;

   // Get our rgb components
   float H = _rt3 * tan( ( h - x ) * _pi2 ) * 3;
   r = ( 1 - s ) * i;
   b = ( ( 3 + H ) * i - ( 1 + H ) * r ) / 2;
   g = 3 * i - b - r;

   // Put them in right order

   if ( h3 < 1 )
   {
      ret = float4( g, b, r, src.a );
   }
   else if ( h3 < 2 )
   {
      ret = float4( r, g, b, src.a );
   }
   else
   {
      ret = float4( b, r, g, src.a );
   }

   return ret;
}
//--------------------------------------------------------------//
DeclareEntryPoint( CompositeOne )
{
   float4 source     = ReadPixel( Input, uv1 );
   float4 corrected1 = tex2D( Inside, uv2 );

   float4 ret = lerp( source, corrected1, corrected1.a );
   ret.a = 1.0;

   return ret;
}

//--------------------------------------------------------------//
DeclareEntryPoint( CompositeTwo_mask )
{
   float4 source     = ReadPixel( Input, uv1 );
   float4 corrected1 = tex2D( Inside, uv2 );
   float4 corrected2 = tex2D( Outside, uv3 );

   float4 ret = lerp( corrected2, corrected1, tex2D( Mask, uv4 ));
   ret.a = source.a;
   return ret;
}
