// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause
/**
 * @class   vtkGlobFileNames
 * @brief   find files that match a wildcard pattern
 *
 * vtkGlobFileNames is a utility for finding files and directories
 * that match a given wildcard pattern. This class supports standard
 * shell-style wildcard patterns for flexible file system searches.
 *
 * Supported wildcard patterns:
 * - "*" matches any sequence of characters (including none)
 * - "?" matches exactly one character
 * - "[abc]" matches any single character from the enclosed set
 * - "[a-z]" matches any single character in the specified range
 * - "[!abc]" matches any single character NOT in the enclosed set
 *
 * Path separators ("/" on Unix, "\" on Windows) receive special treatment
 * and are never matched by wildcards. They can only be matched by explicit
 * path separators in the pattern, ensuring that wildcard matching respects
 * directory boundaries.
 *
 * @warning
 * This function performs case-sensitive matches on UNIX and
 * case-insensitive matches on Windows.
 *
 * @sa
 * vtkDirectory
 */

#ifndef vtkGlobFileNames_h
#define vtkGlobFileNames_h

#include "vtkIOCoreModule.h" // For export macro
#include "vtkObject.h"

VTK_ABI_NAMESPACE_BEGIN
class vtkStringArray;

class VTKIOCORE_EXPORT vtkGlobFileNames : public vtkObject
{
public:
  ///@{
  /**
   * Return the class name as a string.
   */
  vtkTypeMacro(vtkGlobFileNames, vtkObject);
  ///@}

  /**
   * Create a new vtkGlobFileNames object.
   */
  static vtkGlobFileNames* New();

  /**
   * Print directory to stream.
   */
  void PrintSelf(ostream& os, vtkIndent indent) override;

  /**
   * Reset the glob by clearing the list of output filenames.
   */
  void Reset();

  ///@{
  /**
   * Set the directory in which to perform the glob.  If this is
   * not set, then the current directory will be used.  Also, if
   * you use a glob pattern that contains absolute path (one that
   * starts with "/" or a drive letter) then that absolute path
   * will be used and Directory will be ignored.
   */
  vtkSetFilePathMacro(Directory);
  vtkGetFilePathMacro(Directory);
  ///@}

  /**
   * Search for all files that match the given expression,
   * sort them, and add them to the output.  This method can
   * be called repeatedly to add files matching additional patterns.
   * Returns 1 if successful, otherwise returns zero.
   */
  int AddFileNames(VTK_FILEPATH const char* pattern);

  ///@{
  /**
   * Recurse into subdirectories.
   */
  vtkSetMacro(Recurse, vtkTypeBool);
  vtkBooleanMacro(Recurse, vtkTypeBool);
  vtkGetMacro(Recurse, vtkTypeBool);
  ///@}

  /**
   * Return the number of files found.
   */
  int GetNumberOfFileNames();

  /**
   * Return the file at the given index, the indexing is 0 based.
   */
  VTK_FILEPATH const char* GetNthFileName(int index);

  ///@{
  /**
   * Get an array that contains all the file names.
   */
  vtkGetObjectMacro(FileNames, vtkStringArray);
  ///@}

protected:
  ///@{
  /**
   * Set the wildcard pattern.
   */
  vtkSetFilePathMacro(Pattern);
  vtkGetFilePathMacro(Pattern);
  ///@}

  vtkGlobFileNames();
  ~vtkGlobFileNames() override;

private:
  char* Directory;           // Directory for search.
  char* Pattern;             // Wildcard pattern
  vtkTypeBool Recurse;       // Recurse into subdirectories
  vtkStringArray* FileNames; // VTK array of files

  vtkGlobFileNames(const vtkGlobFileNames&) = delete;
  void operator=(const vtkGlobFileNames&) = delete;
};

VTK_ABI_NAMESPACE_END
#endif
