﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/servicediscovery/ServiceDiscoveryRequest.h>
#include <aws/servicediscovery/ServiceDiscovery_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ServiceDiscovery {
namespace Model {

/**
 */
class DiscoverInstancesRevisionRequest : public ServiceDiscoveryRequest {
 public:
  AWS_SERVICEDISCOVERY_API DiscoverInstancesRevisionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DiscoverInstancesRevision"; }

  AWS_SERVICEDISCOVERY_API Aws::String SerializePayload() const override;

  AWS_SERVICEDISCOVERY_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The <code>HttpName</code> name of the namespace. The <code>HttpName</code> is
   * found in the <code>HttpProperties</code> member of the <code>Properties</code>
   * member of the namespace.</p>
   */
  inline const Aws::String& GetNamespaceName() const { return m_namespaceName; }
  inline bool NamespaceNameHasBeenSet() const { return m_namespaceNameHasBeenSet; }
  template <typename NamespaceNameT = Aws::String>
  void SetNamespaceName(NamespaceNameT&& value) {
    m_namespaceNameHasBeenSet = true;
    m_namespaceName = std::forward<NamespaceNameT>(value);
  }
  template <typename NamespaceNameT = Aws::String>
  DiscoverInstancesRevisionRequest& WithNamespaceName(NamespaceNameT&& value) {
    SetNamespaceName(std::forward<NamespaceNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the service that you specified when you registered the
   * instance.</p>
   */
  inline const Aws::String& GetServiceName() const { return m_serviceName; }
  inline bool ServiceNameHasBeenSet() const { return m_serviceNameHasBeenSet; }
  template <typename ServiceNameT = Aws::String>
  void SetServiceName(ServiceNameT&& value) {
    m_serviceNameHasBeenSet = true;
    m_serviceName = std::forward<ServiceNameT>(value);
  }
  template <typename ServiceNameT = Aws::String>
  DiscoverInstancesRevisionRequest& WithServiceName(ServiceNameT&& value) {
    SetServiceName(std::forward<ServiceNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the Amazon Web Services account that owns the namespace associated
   * with the instance, as specified in the namespace <code>ResourceOwner</code>
   * field. For instances associated with namespaces that are shared with your
   * account, you must specify an <code>OwnerAccount</code>. For more information
   * about shared namespaces, see <a
   * href="https://docs.aws.amazon.com/cloud-map/latest/dg/sharing-namespaces.html">Cross-account
   * Cloud Map namespace sharing</a> in the <i>Cloud Map Developer Guide</i>.</p>
   */
  inline const Aws::String& GetOwnerAccount() const { return m_ownerAccount; }
  inline bool OwnerAccountHasBeenSet() const { return m_ownerAccountHasBeenSet; }
  template <typename OwnerAccountT = Aws::String>
  void SetOwnerAccount(OwnerAccountT&& value) {
    m_ownerAccountHasBeenSet = true;
    m_ownerAccount = std::forward<OwnerAccountT>(value);
  }
  template <typename OwnerAccountT = Aws::String>
  DiscoverInstancesRevisionRequest& WithOwnerAccount(OwnerAccountT&& value) {
    SetOwnerAccount(std::forward<OwnerAccountT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_namespaceName;
  bool m_namespaceNameHasBeenSet = false;

  Aws::String m_serviceName;
  bool m_serviceNameHasBeenSet = false;

  Aws::String m_ownerAccount;
  bool m_ownerAccountHasBeenSet = false;
};

}  // namespace Model
}  // namespace ServiceDiscovery
}  // namespace Aws
