﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/waf/WAFRequest.h>
#include <aws/waf/WAF_EXPORTS.h>
#include <aws/waf/model/RuleUpdate.h>

#include <utility>

namespace Aws {
namespace WAF {
namespace Model {

/**
 */
class UpdateRateBasedRuleRequest : public WAFRequest {
 public:
  AWS_WAF_API UpdateRateBasedRuleRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateRateBasedRule"; }

  AWS_WAF_API Aws::String SerializePayload() const override;

  AWS_WAF_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The <code>RuleId</code> of the <code>RateBasedRule</code> that you want to
   * update. <code>RuleId</code> is returned by <code>CreateRateBasedRule</code> and
   * by <a>ListRateBasedRules</a>.</p>
   */
  inline const Aws::String& GetRuleId() const { return m_ruleId; }
  inline bool RuleIdHasBeenSet() const { return m_ruleIdHasBeenSet; }
  template <typename RuleIdT = Aws::String>
  void SetRuleId(RuleIdT&& value) {
    m_ruleIdHasBeenSet = true;
    m_ruleId = std::forward<RuleIdT>(value);
  }
  template <typename RuleIdT = Aws::String>
  UpdateRateBasedRuleRequest& WithRuleId(RuleIdT&& value) {
    SetRuleId(std::forward<RuleIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The value returned by the most recent call to <a>GetChangeToken</a>.</p>
   */
  inline const Aws::String& GetChangeToken() const { return m_changeToken; }
  inline bool ChangeTokenHasBeenSet() const { return m_changeTokenHasBeenSet; }
  template <typename ChangeTokenT = Aws::String>
  void SetChangeToken(ChangeTokenT&& value) {
    m_changeTokenHasBeenSet = true;
    m_changeToken = std::forward<ChangeTokenT>(value);
  }
  template <typename ChangeTokenT = Aws::String>
  UpdateRateBasedRuleRequest& WithChangeToken(ChangeTokenT&& value) {
    SetChangeToken(std::forward<ChangeTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An array of <code>RuleUpdate</code> objects that you want to insert into or
   * delete from a <a>RateBasedRule</a>. </p>
   */
  inline const Aws::Vector<RuleUpdate>& GetUpdates() const { return m_updates; }
  inline bool UpdatesHasBeenSet() const { return m_updatesHasBeenSet; }
  template <typename UpdatesT = Aws::Vector<RuleUpdate>>
  void SetUpdates(UpdatesT&& value) {
    m_updatesHasBeenSet = true;
    m_updates = std::forward<UpdatesT>(value);
  }
  template <typename UpdatesT = Aws::Vector<RuleUpdate>>
  UpdateRateBasedRuleRequest& WithUpdates(UpdatesT&& value) {
    SetUpdates(std::forward<UpdatesT>(value));
    return *this;
  }
  template <typename UpdatesT = RuleUpdate>
  UpdateRateBasedRuleRequest& AddUpdates(UpdatesT&& value) {
    m_updatesHasBeenSet = true;
    m_updates.emplace_back(std::forward<UpdatesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of requests, which have an identical value in the field
   * specified by the <code>RateKey</code>, allowed in a five-minute period. If the
   * number of requests exceeds the <code>RateLimit</code> and the other predicates
   * specified in the rule are also met, AWS WAF triggers the action that is
   * specified for this rule.</p>
   */
  inline long long GetRateLimit() const { return m_rateLimit; }
  inline bool RateLimitHasBeenSet() const { return m_rateLimitHasBeenSet; }
  inline void SetRateLimit(long long value) {
    m_rateLimitHasBeenSet = true;
    m_rateLimit = value;
  }
  inline UpdateRateBasedRuleRequest& WithRateLimit(long long value) {
    SetRateLimit(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_ruleId;

  Aws::String m_changeToken;

  Aws::Vector<RuleUpdate> m_updates;

  long long m_rateLimit{0};
  bool m_ruleIdHasBeenSet = false;
  bool m_changeTokenHasBeenSet = false;
  bool m_updatesHasBeenSet = false;
  bool m_rateLimitHasBeenSet = false;
};

}  // namespace Model
}  // namespace WAF
}  // namespace Aws
