﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/dms/DatabaseMigrationService_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace DatabaseMigrationService {
namespace Model {

/**
 * <p>Provides information about the limitations of target Amazon Web Services
 * engines.</p> <p>Your source database might include features that the target
 * Amazon Web Services engine doesn't support. Fleet Advisor lists these features
 * as limitations. You should consider these limitations during database migration.
 * For each limitation, Fleet Advisor recommends an action that you can take to
 * address or avoid this limitation.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/dms-2016-01-01/Limitation">AWS API
 * Reference</a></p>
 */
class Limitation {
 public:
  AWS_DATABASEMIGRATIONSERVICE_API Limitation() = default;
  AWS_DATABASEMIGRATIONSERVICE_API Limitation(Aws::Utils::Json::JsonView jsonValue);
  AWS_DATABASEMIGRATIONSERVICE_API Limitation& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_DATABASEMIGRATIONSERVICE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The identifier of the source database.</p>
   */
  inline const Aws::String& GetDatabaseId() const { return m_databaseId; }
  inline bool DatabaseIdHasBeenSet() const { return m_databaseIdHasBeenSet; }
  template <typename DatabaseIdT = Aws::String>
  void SetDatabaseId(DatabaseIdT&& value) {
    m_databaseIdHasBeenSet = true;
    m_databaseId = std::forward<DatabaseIdT>(value);
  }
  template <typename DatabaseIdT = Aws::String>
  Limitation& WithDatabaseId(DatabaseIdT&& value) {
    SetDatabaseId(std::forward<DatabaseIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the target engine that Fleet Advisor should use in the target
   * engine recommendation. Valid values include <code>"rds-aurora-mysql"</code>,
   * <code>"rds-aurora-postgresql"</code>, <code>"rds-mysql"</code>,
   * <code>"rds-oracle"</code>, <code>"rds-sql-server"</code>, and
   * <code>"rds-postgresql"</code>.</p>
   */
  inline const Aws::String& GetEngineName() const { return m_engineName; }
  inline bool EngineNameHasBeenSet() const { return m_engineNameHasBeenSet; }
  template <typename EngineNameT = Aws::String>
  void SetEngineName(EngineNameT&& value) {
    m_engineNameHasBeenSet = true;
    m_engineName = std::forward<EngineNameT>(value);
  }
  template <typename EngineNameT = Aws::String>
  Limitation& WithEngineName(EngineNameT&& value) {
    SetEngineName(std::forward<EngineNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the limitation. Describes unsupported database features,
   * migration action items, and other limitations.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  Limitation& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the limitation. Provides additional information about the
   * limitation, and includes recommended actions that you can take to address or
   * avoid this limitation.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  Limitation& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The impact of the limitation. You can use this parameter to prioritize
   * limitations that you want to address. Valid values include
   * <code>"Blocker"</code>, <code>"High"</code>, <code>"Medium"</code>, and
   * <code>"Low"</code>.</p>
   */
  inline const Aws::String& GetImpact() const { return m_impact; }
  inline bool ImpactHasBeenSet() const { return m_impactHasBeenSet; }
  template <typename ImpactT = Aws::String>
  void SetImpact(ImpactT&& value) {
    m_impactHasBeenSet = true;
    m_impact = std::forward<ImpactT>(value);
  }
  template <typename ImpactT = Aws::String>
  Limitation& WithImpact(ImpactT&& value) {
    SetImpact(std::forward<ImpactT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of the limitation, such as action required, upgrade required, and
   * limited feature.</p>
   */
  inline const Aws::String& GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  template <typename TypeT = Aws::String>
  void SetType(TypeT&& value) {
    m_typeHasBeenSet = true;
    m_type = std::forward<TypeT>(value);
  }
  template <typename TypeT = Aws::String>
  Limitation& WithType(TypeT&& value) {
    SetType(std::forward<TypeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_databaseId;

  Aws::String m_engineName;

  Aws::String m_name;

  Aws::String m_description;

  Aws::String m_impact;

  Aws::String m_type;
  bool m_databaseIdHasBeenSet = false;
  bool m_engineNameHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_impactHasBeenSet = false;
  bool m_typeHasBeenSet = false;
};

}  // namespace Model
}  // namespace DatabaseMigrationService
}  // namespace Aws
